import { View, Text, ScrollView, Pressable, StyleSheet } from "react-native";
import React, { useEffect, useState } from "react";
import { useDispatch, useSelector } from "react-redux";
import BackNavigation from "../../../Utilities/CustomHeader/BackNavigation";
import { ProfileStyles } from "../../ProfileStyle";
import { OrderDetailsStyle } from "./OrderDetailsStyle";
import DotIcon from "../../../../assets/svgs/order details/dot.svg";
import dpr from "../../../Utilities/CustomStyleAttribute/dpr";
import useAuth from "../../../../hooks/useAuth";
import { getOrderDetails } from "../../../../redux/slices/user/orderDetails/orderDetails";
import OrderDetailsSkeleton from "../../../../src/skeletons/screens/profile/order/OrderDetails/OrderDetailsSkeleton";
import { TouchableOpacity } from "react-native";
import useProcessPrice from "../../../../hooks/useProcessPrice";
import { BASE_API_URL_STG } from "@env";
import useLangTranslation from "../../../../hooks/useLangTranslation";
import { PRODUCT_DETAILS } from "../../../../components/Navigation/RouteNames";
import CustomSVG from "../../../../src/components/CustomSVG/CustomSVG";
import { splitCamelCase } from "../../../../helper/splitCamelCase";

const OrderDetails = (props) => {
    const { trans } = useLangTranslation();
    const { itemId } = props?.route?.params;
    const dispatch = useDispatch();
    const { access_token } = useAuth();
    const processPrice = useProcessPrice();
    const [inFileLoading, setInFileLoading] = useState(true);
    const [estimateDelivery, setEstimateDelivery] = useState(null);
    const { orderDetails, loading } = useSelector(
        (state) => state.getOrderDetailsSlice || {}
    );
    const { countries } = useSelector(
        (state) => state.getCountries
    );
    const [countryInfo, setCountryInfo] = useState(null);

    const navigateToHome = props.route?.params?.resetToHome
    const {
        created_at = "",
        shipping,
        discount_total,
        shipping_total,
        shipping_title,
        order_key,
        line_items, 
        sub_total,
        total_tax,
        total,
        status,
        payment_status,
        payment
    } = orderDetails || {};

    useEffect(() => {
        (async () => {
            let isMounted = true;
            if (isMounted) {
                const URL = `${BASE_API_URL_STG}/user/orders/${itemId}`;
                await dispatch(getOrderDetails({ access_token, URL }));
                setInFileLoading(false);
            }
            return () => {
                isMounted = false;
            };
        })()
    }, []);

    useEffect(() => {
        const countryInfoNew = countries?.find(
            (country) => country.iso2 == shipping?.country
        );
        setCountryInfo(countryInfoNew);
    }, [shipping?.country, countries])

    useEffect(() => {
        if (navigateToHome) {
          props.navigation.addListener('beforeRemove', e => {
            e.preventDefault();
            props.navigation.dispatch(props.navigation.navigate("Home"));
          });
        }
      }, [props.navigation, navigateToHome]);

    const redirectToPayment = () => {
        props.navigation.push("payment webview", {
            message: "Payment",
            itemId,
            url: orderDetails.payment_link,
        });
    };

    useEffect(() => {
        const calculateEstimateDelivery = (orders) => {
            let filteredOrders = orders?.filter(
                (order) => order.estimate_delivery !== null
            );
            if (filteredOrders.length === 0) return null;
            else if(filteredOrders.length > 0) {
                let min = filteredOrders[0]?.estimate_delivery;
                let max = filteredOrders[0]?.estimate_delivery;
                for (let i = 1; i < filteredOrders?.length; i++) {
                    if (filteredOrders[i].estimate_delivery < min) {
                        min = filteredOrders[i].estimate_delivery;
                    }
                    if (filteredOrders[i].estimate_delivery > max) {
                        max = filteredOrders[i].estimate_delivery;
                    }
                }
                if (min === max) {
                    return min;
                }
                return `${min} - ${max}`;
            }
        };

        if (line_items?.length) {
            const result = calculateEstimateDelivery(line_items);
            setEstimateDelivery(result);
        }
    }, [line_items?.length]);

    const [fastLoad, setFastLoad] = useState(true);
    useEffect(() => setFastLoad(false), []);
    if (fastLoad) return <View />;

    const styles = Styles(line_items, status);

    return (
        <>
            <BackNavigation
                navigationProps={props.navigation}
                routeName={trans("order details")}
                capitalize={true}
            />
            <View style={ProfileStyles.hrLine} />
            <View style={OrderDetailsStyle.container}>
                {inFileLoading || loading ? (
                    <OrderDetailsSkeleton />
                ) : (
                    <ScrollView showsVerticalScrollIndicator={false}>
                        <View>
                            <Text style={OrderDetailsStyle.idNum}>
                                {trans("ID Number")}
                            </Text>
                            <Text style={OrderDetailsStyle.invoiceId}>
                                #{order_key}
                            </Text>
                            <View style={OrderDetailsStyle.statusCont}>
                                <View
                                    style={[
                                        OrderDetailsStyle.status,
                                        styles.statusBackground,
                                    ]}
                                >
                                    <CustomSVG svgIcon={DotIcon} fill={dynamicStatusText(status)} />
                                    <Text
                                        style={[
                                            OrderDetailsStyle.statusText,
                                            styles.statusTextColor,
                                        ]}
                                    >
                                        {trans(status)}
                                    </Text>
                                </View>
                            </View>
                        </View>
                        <View style={OrderDetailsStyle.infoCont}>
                            <View style={OrderDetailsStyle.deliveryCont}>
                            {shipping_title && <View style={OrderDetailsStyle.deliverySubCont}>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Shipping Method")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {trans(shipping_title)}
                                    </Text>
                                </View>}
                                <View
                                    style={[
                                        OrderDetailsStyle.deliverySubCont,
                                        shipping_title && styles.paymentStatusContainer
                                    ]}
                                >
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Payment Status")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {payment_status && trans(payment_status)}
                                    </Text>
                                </View>
                            </View>
                            <View
                                style={[
                                    OrderDetailsStyle.deliveryCont,
                                    styles.deliveryContainer
                                ]}
                            >
                                <View style={OrderDetailsStyle.deliverySubCont}>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Order Date")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {created_at}
                                    </Text>
                                </View>
                                {estimateDelivery != null ? (
                                    <View
                                        style={[
                                            OrderDetailsStyle.deliverySubCont,
                                            styles.deliverySubContainer
                                        ]}
                                    >
                                        <Text
                                            style={
                                                OrderDetailsStyle.deliveryText1
                                            }
                                        >
                                            {trans("Estimated Delivery")}
                                        </Text>
                                        <Text
                                            style={
                                                OrderDetailsStyle.deliveryText2
                                            }
                                        >
                                            {trans("{{x}} days (all items)", {
                                                x: estimateDelivery,
                                            })}
                                        </Text>
                                    </View>
                                ) : payment?.gateway ?
                                    <View style={[
                                        OrderDetailsStyle.deliverySubCont,
                                        styles.deliverySubContainer
                                    ]}>
                                        <Text
                                            style={[OrderDetailsStyle.deliveryText1]}
                                        >
                                            {trans("Payment Method")}
                                        </Text>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText2}
                                        >
                                            {splitCamelCase(payment?.gateway)}
                                        </Text>
                                    </View>
                                    : null
                                }
                            </View>
                            {(estimateDelivery != null && payment?.gateway) &&
                                <View
                                    style={[
                                        OrderDetailsStyle.deliveryCont,
                                        styles.deliveryContainer
                                    ]}
                                >
                                    <View style={OrderDetailsStyle.deliverySubCont}>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText1}
                                        >
                                            {trans("Payment Method")}
                                        </Text>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText2}
                                        >
                                            {splitCamelCase(payment?.gateway)}
                                        </Text>
                                    </View>
                                </View>
                            }
                            <View>
                                <Text style={OrderDetailsStyle.shoppingTitle}>
                                    {trans("Shipping Address")}
                                </Text>
                                <View
                                    style={OrderDetailsStyle.mt_15}
                                >
                                    {shipping?.address1 &&
                                        <View
                                            style={
                                                OrderDetailsStyle.shoppingAddressCont
                                            }
                                        >
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {trans("Street")}
                                            </Text>
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {shipping.address1}
                                            </Text>
                                        </View>
                                    }
                                    {shipping?.city &&
                                        <View
                                            style={
                                                OrderDetailsStyle.shoppingAddressCont
                                            }
                                        >
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {trans("City")}
                                            </Text>
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {shipping.city}
                                            </Text>
                                        </View>
                                    }
                                    {shipping?.post_code &&
                                        <View
                                            style={
                                                OrderDetailsStyle.shoppingAddressCont
                                            }
                                        >
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {trans("Postcode")}
                                            </Text>
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {shipping.post_code}
                                            </Text>
                                        </View>
                                    }
                                    {shipping?.country &&
                                        <View
                                            style={
                                                OrderDetailsStyle.shoppingAddressCont
                                            }
                                        >
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {trans("Country")}
                                            </Text>
                                            <Text
                                                style={[
                                                    OrderDetailsStyle.shoppingText,
                                                    OrderDetailsStyle.shoppingTextWidth
                                                ]}
                                            >
                                                {countryInfo?.name || shipping.country}
                                            </Text>
                                        </View>
                                    }
                                </View>
                            </View>
                            <View style={OrderDetailsStyle.orderProductsCont}>
                                <Text
                                    style={OrderDetailsStyle.orderProductTitle}
                                >
                                    {trans("Ordered Products")}
                                </Text>
                                {line_items?.map((item, index) => (
                                    <View key={`key` + index}>
                                        <Pressable
                                            onPress={() => {
                                                props.navigation.navigate(
                                                    PRODUCT_DETAILS,
                                                    {
                                                        slug: item?.slug,
                                                    }
                                                );
                                            }}
                                            style={[
                                                OrderDetailsStyle.singleProductCont,
                                                styles.singleProductContainer(index)
                                            ]}
                                        >
                                            <View>
                                                <Text
                                                    style={
                                                        OrderDetailsStyle.name
                                                    }
                                                >
                                                    {item?.name.slice(0, 20)} .
                                                    . .
                                                </Text>
                                                <View
                                                    style={
                                                        OrderDetailsStyle.items
                                                    }
                                                >
                                                    <Text
                                                        style={
                                                            OrderDetailsStyle.quantity
                                                        }
                                                    >
                                                        {item?.quantity} x {trans("Item")}
                                                    </Text>
                                                    <View
                                                        style={
                                                            OrderDetailsStyle.vrLine
                                                        }
                                                    />
                                                    <Text
                                                        style={
                                                            OrderDetailsStyle.quantity
                                                        }
                                                    >
                                                        {
                                                            item?.vendor?.shop
                                                                ?.name
                                                        }
                                                    </Text>
                                                </View>
                                            </View>
                                            <View>
                                                <Text
                                                    style={
                                                        OrderDetailsStyle.name
                                                    }
                                                >
                                                    {processPrice(
                                                        parseInt(item?.price) *
                                                            parseInt(
                                                                item?.quantity
                                                            )
                                                    )}
                                                </Text>
                                                <Text
                                                    style={
                                                        OrderDetailsStyle.quantity
                                                    }
                                                >
                                                    {item?.is_delivered
                                                        ? trans("Delivered")
                                                        : trans("Pending")}
                                                </Text>
                                            </View>
                                        </Pressable>
                                        {line_items.length !== index + 1 && (
                                            <View
                                                style={OrderDetailsStyle.hrLine}
                                            />
                                        )}
                                    </View>
                                ))}
                            </View>
                            <View style={OrderDetailsStyle.calculationMargin}>
                                <View style={OrderDetailsStyle.subTotalCont}>
                                    <Text style={OrderDetailsStyle.subText}>
                                        {trans("Subtotal")}
                                    </Text>
                                    <Text style={OrderDetailsStyle.subText}>
                                        {processPrice(sub_total)}
                                    </Text>
                                </View>
                                <View style={OrderDetailsStyle.subTotalCont}>
                                    <Text style={OrderDetailsStyle.subText}>
                                        {trans("Shipping")}
                                    </Text>
                                    <Text style={OrderDetailsStyle.subText}>
                                        {processPrice(shipping_total)}
                                    </Text>
                                </View>
                                <View
                                    style={[
                                        OrderDetailsStyle.subTotalCont,
                                        discount_total &&
                                            OrderDetailsStyle.taxCont,
                                    ]}
                                >
                                    <Text style={OrderDetailsStyle.subText}>
                                        {trans("Tax")}
                                    </Text>
                                    <Text style={OrderDetailsStyle.subText}>
                                        {processPrice(total_tax)}
                                    </Text>
                                </View>
                                {discount_total > 0 && (
                                    <View
                                        style={[
                                            OrderDetailsStyle.subTotalCont,
                                            OrderDetailsStyle.taxCont,
                                        ]}
                                    >
                                        <Text style={OrderDetailsStyle.subText}>
                                            {trans("Discount")}
                                        </Text>
                                        <Text style={OrderDetailsStyle.subText}>
                                            {processPrice(discount_total)}
                                        </Text>
                                    </View>
                                )}
                                <View
                                    style={[
                                        OrderDetailsStyle.subTotalCont,
                                        OrderDetailsStyle.grandTotal,
                                    ]}
                                >
                                    <Text
                                        style={[
                                            OrderDetailsStyle.subText,
                                            styles.subTextColor,
                                        ]}
                                    >
                                        {trans("Grand Total")}
                                    </Text>
                                    <Text
                                        style={[
                                            OrderDetailsStyle.subText,
                                            styles.subTextColor,
                                        ]}
                                    >
                                        {processPrice(total)}
                                    </Text>
                                </View>

                                {(status == "Pending Payment" || status == "Refunded" || status == "On hold" || status == "Failed" || status == "Unpaid") && orderDetails.payment_link && payment?.gateway != "CashOnDelivery" && payment?.gateway != "DirectBankTransfer" && (
                                    <TouchableOpacity
                                        style={[
                                            OrderDetailsStyle.subTotalCont,
                                            OrderDetailsStyle.grandTotal,
                                        ]}
                                        onPress={redirectToPayment}
                                    >
                                        <Text
                                            style={[
                                                OrderDetailsStyle.subText,
                                                styles.subTextColor
                                            ]}
                                        >
                                            {trans("Proceed to payment")}
                                        </Text>
                                    </TouchableOpacity>
                                )}
                            </View>
                        </View>
                    </ScrollView>
                )}
            </View>
        </>
    );
};

export default OrderDetails;

const Styles = (line_items, status) => StyleSheet.create({
    singleProductContainer: (index) => ({
        borderBottomColor:
            index ===
            line_items?.length - 1
                ? "transparent"
                : "#DFDFDF",
    }),
    subTextColor: { color: "#FFFFFF" },
    statusBackground: {
        backgroundColor: dynamicStatusBG(status)
    },
    statusTextColor: {
        color: dynamicStatusText(
            status
        ),
    },
    paymentStatusContainer: {
        borderLeftWidth: 1,
        borderColor: "#B1B1B1",
    },
    deliveryContainer: {
        marginTop: dpr(15)
    },
    deliverySubContainer: {
        borderLeftWidth: 1,
        borderColor: "#B1B1B1"
    }
});

const dynamicStatusText = (status) => {
    return status === "Pending Payment"
        ? "#898989"
        : status === "Failed"
        ? "#C8191C"
        : status === "Processing"
        ? "#DEA512"
        : status === "Completed"
        ? "#009651"
        : status === "On hold"
        ? "#FD7F0C"
        : status === "Cancelled"
        ? "#C8191C"
        : status === "Refunded"
        ? "#009651"
        : "#000000";
};
const dynamicStatusBG = (status) => {
    return status === "Pending Payment"
        ? "#F4F4F4"
        : status === "Failed"
        ? "#F9E8E8"
        : status === "Processing"
        ? "#FEF8E7"
        : status === "Completed"
        ? "#EBF9F1"
        : status === "On hold"
        ? "#FEF2E7"
        : status === "Cancelled"
        ? "#F9E8E8"
        : status === "Refunded"
        ? "#EBF9F1"
        : "#FFFFFF";
};
